﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/DnsNameState.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Information about the private DNS name for the service
 * endpoint.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/PrivateDnsNameConfiguration">AWS
 * API Reference</a></p>
 */
class PrivateDnsNameConfiguration {
 public:
  AWS_EC2_API PrivateDnsNameConfiguration() = default;
  AWS_EC2_API PrivateDnsNameConfiguration(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API PrivateDnsNameConfiguration& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The verification state of the VPC endpoint service.</p> <p>Consumers of the
   * endpoint service can use the private name only when the state is
   * <code>verified</code>.</p>
   */
  inline DnsNameState GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  inline void SetState(DnsNameState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline PrivateDnsNameConfiguration& WithState(DnsNameState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The endpoint service verification type, for example TXT.</p>
   */
  inline const Aws::String& GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  template <typename TypeT = Aws::String>
  void SetType(TypeT&& value) {
    m_typeHasBeenSet = true;
    m_type = std::forward<TypeT>(value);
  }
  template <typename TypeT = Aws::String>
  PrivateDnsNameConfiguration& WithType(TypeT&& value) {
    SetType(std::forward<TypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value the service provider adds to the private DNS name domain record
   * before verification.</p>
   */
  inline const Aws::String& GetValue() const { return m_value; }
  inline bool ValueHasBeenSet() const { return m_valueHasBeenSet; }
  template <typename ValueT = Aws::String>
  void SetValue(ValueT&& value) {
    m_valueHasBeenSet = true;
    m_value = std::forward<ValueT>(value);
  }
  template <typename ValueT = Aws::String>
  PrivateDnsNameConfiguration& WithValue(ValueT&& value) {
    SetValue(std::forward<ValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the record subdomain the service provider needs to create. The
   * service provider adds the <code>value</code> text to the <code>name</code>.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  PrivateDnsNameConfiguration& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}
 private:
  DnsNameState m_state{DnsNameState::NOT_SET};

  Aws::String m_type;

  Aws::String m_value;

  Aws::String m_name;
  bool m_stateHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_valueHasBeenSet = false;
  bool m_nameHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
