﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codecommit/CodeCommitRequest.h>
#include <aws/codecommit/CodeCommit_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeCommit {
namespace Model {

/**
 * <p>Represents the input of a get blob operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetBlobInput">AWS
 * API Reference</a></p>
 */
class GetBlobRequest : public CodeCommitRequest {
 public:
  AWS_CODECOMMIT_API GetBlobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetBlob"; }

  AWS_CODECOMMIT_API Aws::String SerializePayload() const override;

  AWS_CODECOMMIT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the repository that contains the blob.</p>
   */
  inline const Aws::String& GetRepositoryName() const { return m_repositoryName; }
  inline bool RepositoryNameHasBeenSet() const { return m_repositoryNameHasBeenSet; }
  template <typename RepositoryNameT = Aws::String>
  void SetRepositoryName(RepositoryNameT&& value) {
    m_repositoryNameHasBeenSet = true;
    m_repositoryName = std::forward<RepositoryNameT>(value);
  }
  template <typename RepositoryNameT = Aws::String>
  GetBlobRequest& WithRepositoryName(RepositoryNameT&& value) {
    SetRepositoryName(std::forward<RepositoryNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the blob, which is its SHA-1 pointer.</p>
   */
  inline const Aws::String& GetBlobId() const { return m_blobId; }
  inline bool BlobIdHasBeenSet() const { return m_blobIdHasBeenSet; }
  template <typename BlobIdT = Aws::String>
  void SetBlobId(BlobIdT&& value) {
    m_blobIdHasBeenSet = true;
    m_blobId = std::forward<BlobIdT>(value);
  }
  template <typename BlobIdT = Aws::String>
  GetBlobRequest& WithBlobId(BlobIdT&& value) {
    SetBlobId(std::forward<BlobIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_repositoryName;

  Aws::String m_blobId;
  bool m_repositoryNameHasBeenSet = false;
  bool m_blobIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeCommit
}  // namespace Aws
