/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.storage;

import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.connect.runtime.WorkerConfig;
import org.apache.kafka.connect.storage.KafkaOffsetBackingStore;
import org.apache.kafka.connect.storage.OffsetBackingStore;
import org.apache.kafka.connect.util.Callback;
import org.apache.kafka.connect.util.LoggingContext;
import org.apache.kafka.connect.util.TopicAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectorOffsetBackingStore
implements OffsetBackingStore {
    private static final Logger log = LoggerFactory.getLogger(ConnectorOffsetBackingStore.class);
    private final Time time;
    private final Supplier<LoggingContext> loggingContext;
    private final String primaryOffsetsTopic;
    private final Optional<OffsetBackingStore> workerStore;
    private final Optional<KafkaOffsetBackingStore> connectorStore;
    private final Optional<TopicAdmin> connectorStoreAdmin;

    public static ConnectorOffsetBackingStore withConnectorAndWorkerStores(Supplier<LoggingContext> loggingContext, OffsetBackingStore workerStore, KafkaOffsetBackingStore connectorStore, String connectorOffsetsTopic, TopicAdmin connectorStoreAdmin) {
        Objects.requireNonNull(loggingContext);
        Objects.requireNonNull(workerStore);
        Objects.requireNonNull(connectorStore);
        Objects.requireNonNull(connectorOffsetsTopic);
        Objects.requireNonNull(connectorStoreAdmin);
        return new ConnectorOffsetBackingStore(Time.SYSTEM, loggingContext, connectorOffsetsTopic, workerStore, connectorStore, connectorStoreAdmin);
    }

    public static ConnectorOffsetBackingStore withOnlyWorkerStore(Supplier<LoggingContext> loggingContext, OffsetBackingStore workerStore, String workerOffsetsTopic) {
        Objects.requireNonNull(loggingContext);
        Objects.requireNonNull(workerStore);
        return new ConnectorOffsetBackingStore(Time.SYSTEM, loggingContext, workerOffsetsTopic, workerStore, null, null);
    }

    public static ConnectorOffsetBackingStore withOnlyConnectorStore(Supplier<LoggingContext> loggingContext, KafkaOffsetBackingStore connectorStore, String connectorOffsetsTopic, TopicAdmin connectorStoreAdmin) {
        Objects.requireNonNull(loggingContext);
        Objects.requireNonNull(connectorOffsetsTopic);
        Objects.requireNonNull(connectorStoreAdmin);
        return new ConnectorOffsetBackingStore(Time.SYSTEM, loggingContext, connectorOffsetsTopic, null, connectorStore, connectorStoreAdmin);
    }

    ConnectorOffsetBackingStore(Time time, Supplier<LoggingContext> loggingContext, String primaryOffsetsTopic, OffsetBackingStore workerStore, KafkaOffsetBackingStore connectorStore, TopicAdmin connectorStoreAdmin) {
        if (workerStore == null && connectorStore == null) {
            throw new IllegalArgumentException("At least one non-null offset store must be provided");
        }
        this.time = time;
        this.loggingContext = loggingContext;
        this.primaryOffsetsTopic = primaryOffsetsTopic;
        this.workerStore = Optional.ofNullable(workerStore);
        this.connectorStore = Optional.ofNullable(connectorStore);
        this.connectorStoreAdmin = Optional.ofNullable(connectorStoreAdmin);
    }

    public String primaryOffsetsTopic() {
        return this.primaryOffsetsTopic;
    }

    @Override
    public void start() {
        this.connectorStore.ifPresent(OffsetBackingStore::start);
    }

    @Override
    public void stop() {
        this.connectorStore.ifPresent(OffsetBackingStore::stop);
        this.connectorStoreAdmin.ifPresent(TopicAdmin::close);
    }

    @Override
    public Future<Map<ByteBuffer, ByteBuffer>> get(Collection<ByteBuffer> keys) {
        final Future<Map<ByteBuffer, ByteBuffer>> workerGetFuture = ConnectorOffsetBackingStore.getFromStore(this.workerStore, keys);
        final Future<Map<ByteBuffer, ByteBuffer>> connectorGetFuture = ConnectorOffsetBackingStore.getFromStore(this.connectorStore, keys);
        return new Future<Map<ByteBuffer, ByteBuffer>>(){

            @Override
            public boolean cancel(boolean mayInterruptIfRunning) {
                return workerGetFuture.cancel(mayInterruptIfRunning) | connectorGetFuture.cancel(mayInterruptIfRunning);
            }

            @Override
            public boolean isCancelled() {
                return workerGetFuture.isCancelled() || connectorGetFuture.isCancelled();
            }

            @Override
            public boolean isDone() {
                return workerGetFuture.isDone() && connectorGetFuture.isDone();
            }

            @Override
            public Map<ByteBuffer, ByteBuffer> get() throws InterruptedException, ExecutionException {
                HashMap<ByteBuffer, ByteBuffer> result = new HashMap<ByteBuffer, ByteBuffer>((Map)workerGetFuture.get());
                result.putAll((Map)connectorGetFuture.get());
                return result;
            }

            @Override
            public Map<ByteBuffer, ByteBuffer> get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
                long timeoutMs = unit.toMillis(timeout);
                long endTime = ConnectorOffsetBackingStore.this.time.milliseconds() + timeoutMs;
                HashMap<ByteBuffer, ByteBuffer> result = new HashMap<ByteBuffer, ByteBuffer>((Map)workerGetFuture.get(timeoutMs, unit));
                timeoutMs = Math.max(1L, endTime - ConnectorOffsetBackingStore.this.time.milliseconds());
                result.putAll((Map)connectorGetFuture.get(timeoutMs, TimeUnit.MILLISECONDS));
                return result;
            }
        };
    }

    @Override
    public Future<Void> set(Map<ByteBuffer, ByteBuffer> values, Callback<Void> callback) {
        OffsetBackingStore secondaryStore;
        OffsetBackingStore primaryStore;
        if (this.connectorStore.isPresent()) {
            primaryStore = this.connectorStore.get();
            secondaryStore = this.workerStore.orElse(null);
        } else if (this.workerStore.isPresent()) {
            primaryStore = this.workerStore.get();
            secondaryStore = null;
        } else {
            throw new IllegalStateException("At least one non-null offset store must be provided");
        }
        HashMap<ByteBuffer, ByteBuffer> regularOffsets = new HashMap<ByteBuffer, ByteBuffer>();
        HashMap<ByteBuffer, ByteBuffer> tombstoneOffsets = new HashMap<ByteBuffer, ByteBuffer>();
        values.forEach((partition, offset) -> {
            if (offset == null) {
                tombstoneOffsets.put((ByteBuffer)partition, (ByteBuffer)null);
            } else {
                regularOffsets.put((ByteBuffer)partition, (ByteBuffer)offset);
            }
        });
        if (secondaryStore != null && !tombstoneOffsets.isEmpty()) {
            return new ChainedOffsetWriteFuture(primaryStore, secondaryStore, values, regularOffsets, tombstoneOffsets, callback);
        }
        return this.setPrimaryThenSecondary(primaryStore, secondaryStore, values, regularOffsets, callback);
    }

    private Future<Void> setPrimaryThenSecondary(OffsetBackingStore primaryStore, OffsetBackingStore secondaryStore, Map<ByteBuffer, ByteBuffer> completeOffsets, Map<ByteBuffer, ByteBuffer> nonTombstoneOffsets, Callback<Void> callback) {
        return primaryStore.set(completeOffsets, (primaryWriteError, ignored) -> {
            if (secondaryStore != null) {
                if (primaryWriteError != null) {
                    log.trace("Skipping offsets write to secondary store because primary write has failed", primaryWriteError);
                } else {
                    try {
                        secondaryStore.set(nonTombstoneOffsets, (secondaryWriteError, ignored2) -> {
                            try (LoggingContext context = this.loggingContext();){
                                if (secondaryWriteError != null) {
                                    log.warn("Failed to write offsets to secondary backing store", secondaryWriteError);
                                } else {
                                    log.debug("Successfully flushed offsets to secondary backing store");
                                }
                            }
                        });
                    }
                    catch (Exception e) {
                        log.warn("Failed to write offsets to secondary backing store", (Throwable)e);
                    }
                }
            }
            try (LoggingContext context = this.loggingContext();){
                callback.onCompletion(primaryWriteError, (Void)ignored);
            }
        });
    }

    @Override
    public Set<Map<String, Object>> connectorPartitions(String connectorName) {
        HashSet<Map<String, Object>> partitions = new HashSet<Map<String, Object>>();
        this.workerStore.ifPresent(offsetBackingStore -> partitions.addAll(offsetBackingStore.connectorPartitions(connectorName)));
        this.connectorStore.ifPresent(offsetBackingStore -> partitions.addAll(offsetBackingStore.connectorPartitions(connectorName)));
        return partitions;
    }

    @Override
    public void configure(WorkerConfig config) {
        this.connectorStore.ifPresent(store -> store.configure(config));
    }

    public boolean hasConnectorSpecificStore() {
        return this.connectorStore.isPresent();
    }

    public boolean hasWorkerGlobalStore() {
        return this.workerStore.isPresent();
    }

    private LoggingContext loggingContext() {
        LoggingContext result = this.loggingContext.get();
        Objects.requireNonNull(result);
        return result;
    }

    private static Future<Map<ByteBuffer, ByteBuffer>> getFromStore(Optional<? extends OffsetBackingStore> store, Collection<ByteBuffer> keys) {
        return store.map(s -> s.get(keys)).orElseGet(() -> CompletableFuture.completedFuture(Collections.emptyMap()));
    }

    private class ChainedOffsetWriteFuture
    implements Future<Void> {
        private final OffsetBackingStore primaryStore;
        private final OffsetBackingStore secondaryStore;
        private final Map<ByteBuffer, ByteBuffer> completeOffsets;
        private final Map<ByteBuffer, ByteBuffer> regularOffsets;
        private final Callback<Void> callback;
        private final AtomicReference<Throwable> writeError;
        private final CountDownLatch completed;

        public ChainedOffsetWriteFuture(OffsetBackingStore primaryStore, OffsetBackingStore secondaryStore, Map<ByteBuffer, ByteBuffer> completeOffsets, Map<ByteBuffer, ByteBuffer> regularOffsets, Map<ByteBuffer, ByteBuffer> tombstoneOffsets, Callback<Void> callback) {
            this.primaryStore = primaryStore;
            this.secondaryStore = secondaryStore;
            this.completeOffsets = completeOffsets;
            this.regularOffsets = regularOffsets;
            this.callback = callback;
            this.writeError = new AtomicReference();
            this.completed = new CountDownLatch(1);
            secondaryStore.set(tombstoneOffsets, this::onFirstWrite);
        }

        private void onFirstWrite(Throwable error, Void ignored) {
            if (error != null) {
                log.trace("Skipping offsets write to primary store because secondary tombstone write has failed", error);
                try (LoggingContext context = ConnectorOffsetBackingStore.this.loggingContext();){
                    this.callback.onCompletion(error, ignored);
                    this.writeError.compareAndSet(null, error);
                    this.completed.countDown();
                }
                return;
            }
            ConnectorOffsetBackingStore.this.setPrimaryThenSecondary(this.primaryStore, this.secondaryStore, this.completeOffsets, this.regularOffsets, this::onSecondWrite);
        }

        private void onSecondWrite(Throwable error, Void ignored) {
            this.callback.onCompletion(error, ignored);
            this.writeError.compareAndSet(null, error);
            this.completed.countDown();
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            return false;
        }

        @Override
        public boolean isCancelled() {
            return false;
        }

        @Override
        public boolean isDone() {
            return this.completed.getCount() == 0L;
        }

        @Override
        public Void get() throws InterruptedException, ExecutionException {
            this.completed.await();
            if (this.writeError.get() != null) {
                throw new ExecutionException(this.writeError.get());
            }
            return null;
        }

        @Override
        public Void get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
            if (!this.completed.await(timeout, unit)) {
                throw new TimeoutException("Failed to complete offset write in time");
            }
            if (this.writeError.get() != null) {
                throw new ExecutionException(this.writeError.get());
            }
            return null;
        }
    }
}

