/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.tools.consumer;

import java.io.PrintStream;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.regex.Pattern;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.MessageFormatter;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.serialization.Deserializer;
import org.apache.kafka.common.utils.Exit;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.tools.consumer.ConsoleConsumerOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsoleConsumer {
    private static final Logger LOG = LoggerFactory.getLogger(ConsoleConsumer.class);
    private static final CountDownLatch SHUTDOWN_LATCH = new CountDownLatch(1);
    static int messageCount = 0;

    public static void main(String[] args) throws Exception {
        ConsoleConsumerOptions opts = new ConsoleConsumerOptions(args);
        try {
            ConsoleConsumer.run(opts);
        }
        catch (AuthenticationException ae) {
            LOG.error("Authentication failed: terminating consumer process", (Throwable)ae);
            Exit.exit((int)1);
        }
        catch (Throwable t) {
            LOG.error("Unknown error when running consumer: ", t);
            Exit.exit((int)1);
        }
    }

    public static void run(ConsoleConsumerOptions opts) {
        messageCount = 0;
        KafkaConsumer consumer = new KafkaConsumer(opts.consumerProps(), (Deserializer)new ByteArrayDeserializer(), (Deserializer)new ByteArrayDeserializer());
        ConsumerWrapper consumerWrapper = new ConsumerWrapper(opts, (Consumer<byte[], byte[]>)consumer);
        ConsoleConsumer.addShutdownHook(consumerWrapper, opts);
        try {
            ConsoleConsumer.process(opts.maxMessages(), opts.formatter(), consumerWrapper, System.out, opts.skipMessageOnError());
        }
        finally {
            consumerWrapper.cleanup();
            opts.formatter().close();
            ConsoleConsumer.reportRecordCount();
            SHUTDOWN_LATCH.countDown();
        }
    }

    static void addShutdownHook(ConsumerWrapper consumer, ConsoleConsumerOptions conf) {
        Exit.addShutdownHook((String)"consumer-shutdown-hook", () -> {
            try {
                consumer.wakeup();
                SHUTDOWN_LATCH.await();
            }
            catch (Throwable t) {
                LOG.error("Exception while running shutdown hook: ", t);
            }
            if (conf.enableSystestEventsLogging()) {
                System.out.println("shutdown_complete");
            }
        });
    }

    static void process(int maxMessages, MessageFormatter formatter, ConsumerWrapper consumer, PrintStream output, boolean skipMessageOnError) {
        while (messageCount < maxMessages || maxMessages == -1) {
            ConsumerRecord<byte[], byte[]> msg;
            try {
                msg = consumer.receive();
            }
            catch (WakeupException we) {
                LOG.trace("Caught WakeupException because consumer is shutdown, ignore and terminate.");
                return;
            }
            catch (Throwable t) {
                LOG.error("Error processing message, terminating consumer process: ", t);
                return;
            }
            ++messageCount;
            try {
                formatter.writeTo(new ConsumerRecord(msg.topic(), msg.partition(), msg.offset(), msg.timestamp(), msg.timestampType(), 0, 0, (Object)((byte[])msg.key()), (Object)((byte[])msg.value()), msg.headers(), Optional.empty()), output);
            }
            catch (Throwable t) {
                if (skipMessageOnError) {
                    LOG.error("Error processing message, skipping this message: ", t);
                }
                throw t;
            }
            if (!ConsoleConsumer.checkErr(output)) continue;
            return;
        }
    }

    static void reportRecordCount() {
        System.err.println("Processed a total of " + messageCount + " messages");
    }

    static boolean checkErr(PrintStream output) {
        boolean gotError = output.checkError();
        if (gotError) {
            System.err.println("Unable to write to standard out, closing consumer.");
        }
        return gotError;
    }

    public static class ConsumerWrapper {
        final Time time = Time.SYSTEM;
        final long timeoutMs;
        final Consumer<byte[], byte[]> consumer;
        Iterator<ConsumerRecord<byte[], byte[]>> recordIter = Collections.emptyIterator();

        public ConsumerWrapper(ConsoleConsumerOptions opts, Consumer<byte[], byte[]> consumer) {
            this.consumer = consumer;
            this.timeoutMs = opts.timeoutMs();
            Optional<String> topic = opts.topicArg();
            if (topic.isPresent()) {
                if (opts.partitionArg().isPresent()) {
                    this.seek(topic.get(), opts.partitionArg().getAsInt(), opts.offsetArg());
                } else {
                    consumer.subscribe(List.of(topic.get()));
                }
            } else {
                opts.includedTopicsArg().ifPresent(topics -> consumer.subscribe(Pattern.compile(topics)));
            }
        }

        private void seek(String topic, int partitionId, long offset) {
            TopicPartition topicPartition = new TopicPartition(topic, partitionId);
            this.consumer.assign(List.of(topicPartition));
            if (offset == -2L) {
                this.consumer.seekToBeginning(List.of(topicPartition));
            } else if (offset == -1L) {
                this.consumer.seekToEnd(List.of(topicPartition));
            } else {
                this.consumer.seek(topicPartition, offset);
            }
        }

        void resetUnconsumedOffsets() {
            HashMap<TopicPartition, Long> smallestUnconsumedOffsets = new HashMap<TopicPartition, Long>();
            while (this.recordIter.hasNext()) {
                ConsumerRecord<byte[], byte[]> record = this.recordIter.next();
                TopicPartition tp = new TopicPartition(record.topic(), record.partition());
                smallestUnconsumedOffsets.putIfAbsent(tp, record.offset());
            }
            smallestUnconsumedOffsets.forEach((arg_0, arg_1) -> this.consumer.seek(arg_0, arg_1));
        }

        ConsumerRecord<byte[], byte[]> receive() {
            long startTimeMs = this.time.milliseconds();
            while (!this.recordIter.hasNext()) {
                this.recordIter = this.consumer.poll(Duration.ofMillis(this.timeoutMs)).iterator();
                if (this.recordIter.hasNext() || this.time.milliseconds() - startTimeMs <= this.timeoutMs) continue;
                throw new TimeoutException();
            }
            return this.recordIter.next();
        }

        void wakeup() {
            this.consumer.wakeup();
        }

        void cleanup() {
            this.resetUnconsumedOffsets();
            this.consumer.close();
        }
    }
}

