/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.jackrabbit.oak.plugins.document;

import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.jackrabbit.oak.osgi.OsgiUtil;
import org.osgi.service.cm.ConfigurationAdmin;
import org.osgi.service.component.ComponentContext;

import static java.util.Objects.requireNonNull;

/**
 * Configuration for the {@link DocumentNodeStoreService}. Access is provided
 * via {@link Configuration}, while internally the implementation considers
 * entries in the following sequence:
 * <ul>
 *     <li>Framework/system properties, potentially with mapped names. See
 *          {@link #frameworkPropertyNameFor(String)}.
 *     <li>OSGi configuration for {@link DocumentNodeStoreService} with
 *          {@link Configuration#PID} if the property is set via the
 *          OSGi Configuration Admin.
 *     <li>OSGi configuration with {@link Configuration#PRESET_PID}. The
 *          default value for a configuration entry will be provided if the
 *          OSGi Configuration Admin does not have an entry as a preset.
 * </ul>
 */
final class DocumentNodeStoreServiceConfiguration {

    /**
     * Default framework property name prefix.
     */
    private static final String DEFAULT_FWK_PREFIX = "oak.documentstore.";

    /**
     * Name of framework property to configure Mongo Connection URI
     */
    private static final String FWK_PROP_URI = "oak.mongo.uri";

    /**
     * Name of framework property to configure Mongo Database name
     * to use
     */
    private static final String FWK_PROP_DB = "oak.mongo.db";

    /**
     * Name of framework property to configure socket keep-alive for MongoDB
     */
    private static final String FWK_PROP_SO_KEEP_ALIVE = "oak.mongo.socketKeepAlive";

    /**
     * Name of framework property to configure socket timeout for lease update
     * operations on MongoDB.
     */
    private static final String FWK_PROP_MONGO_LEASE_SO_TIMEOUT = "oak.mongo.leaseSocketTimeout";

    /**
     * Name of framework property to configure MongoDB connection pool max size.
     */
    private static final String FWK_PROP_MONGO_MAX_POOL_SIZE = "oak.mongo.maxPoolSize";

    /**
     * Name of framework property to configure MongoDB connection pool min size.
     */
    private static final String FWK_PROP_MONGO_MIN_POOL_SIZE = "oak.mongo.minPoolSize";

    /**
     * Name of framework property to configure MongoDB max connecting.
     */
    private static final String FWK_PROP_MONGO_MAX_CONNECTING = "oak.mongo.maxConnecting";

    /**
     * Name of framework property to configure MongoDB max idle time.
     */
    private static final String FWK_PROP_MONGO_MAX_IDLE_TIME_MILLIS = "oak.mongo.maxIdleTimeMillis";

    /**
     * Name of framework property to configure MongoDB max life time.
     */
    private static final String FWK_PROP_MONGO_MAX_LIFE_TIME_MILLIS = "oak.mongo.maxLifeTimeMillis";

    /**
     * Name of framework property to configure MongoDB connect timeout.
     */
    private static final String FWK_PROP_MONGO_CONNECT_TIMEOUT_MILLIS = "oak.mongo.connectTimeoutMillis";

    /**
     * Name of framework property to configure MongoDB heartbeat frequency.
     */
    private static final String FWK_PROP_MONGO_HEARTBEAT_FREQUENCY_MILLIS = "oak.mongo.heartbeatFrequencyMillis";

    /**
     * Name of framework property to configure MongoDB server selection timeout.
     */
    private static final String FWK_PROP_MONGO_SERVER_SELECTION_TIMEOUT_MILLIS = "oak.mongo.serverSelectionTimeoutMillis";

    /**
     * Name of framework property to configure MongoDB wait queue timeout.
     */
    private static final String FWK_PROP_MONGO_WAIT_QUEUE_TIMEOUT_MILLIS = "oak.mongo.waitQueueTimeoutMillis";

    /**
     * Name of framework property to configure MongoDB socket read timeout.
     */
    private static final String FWK_PROP_MONGO_READ_TIMEOUT_MILLIS = "oak.mongo.readTimeoutMillis";

    /**
     * Name of framework property to configure MongoDB min heartbeat frequency.
     */
    private static final String FWK_PROP_MONGO_MIN_HEARTBEAT_FREQUENCY_MILLIS = "oak.mongo.minHeartbeatFrequencyMillis";

    /**
     * Name of the framework property to configure the update limit.
     */
    private static final String FWK_PROP_UPDATE_LIMIT = "update.limit";

    private static final String PROP_DB = "db";
    private static final String PROP_URI = "mongouri";
    private static final String PROP_HOME = "repository.home";
    static final String PROP_SO_KEEP_ALIVE = "socketKeepAlive";
    static final String PROP_LEASE_SO_TIMEOUT = "leaseSocketTimeout";
    static final String PROP_UPDATE_LIMIT = "updateLimit";
    static final String PROP_MONGO_MAX_POOL_SIZE = "mongoMaxPoolSize";
    static final String PROP_MONGO_MIN_POOL_SIZE = "mongoMinPoolSize";
    static final String PROP_MONGO_MAX_CONNECTING = "mongoMaxConnecting";
    static final String PROP_MONGO_MAX_IDLE_TIME_MILLIS = "mongoMaxIdleTimeMillis";
    static final String PROP_MONGO_MAX_LIFE_TIME_MILLIS = "mongoMaxLifeTimeMillis";
    static final String PROP_MONGO_CONNECT_TIMEOUT_MILLIS = "mongoConnectTimeoutMillis";
    static final String PROP_MONGO_HEARTBEAT_FREQUENCY_MILLIS = "mongoHeartbeatFrequencyMillis";
    static final String PROP_MONGO_SERVER_SELECTION_TIMEOUT_MILLIS = "mongoServerSelectionTimeoutMillis";
    static final String PROP_MONGO_WAIT_QUEUE_TIMEOUT_MILLIS = "mongoWaitQueueTimeoutMillis";
    static final String PROP_MONGO_READ_TIMEOUT_MILLIS = "mongoReadTimeoutMillis";
    static final String PROP_MONGO_MIN_HEARTBEAT_FREQUENCY_MILLIS = "mongoMinHeartbeatFrequencyMillis";

    /**
     * Special mapping of property names to framework properties. All other
     * property names are mapped to framework properties by prefixing them with
     * {@link #DEFAULT_FWK_PREFIX}.
     */
    private static final Map<String, String> FWK_PROP_MAPPING = Map.ofEntries(
            Map.entry(PROP_DB, FWK_PROP_DB),
            Map.entry(PROP_URI, FWK_PROP_URI),
            Map.entry(PROP_HOME, PROP_HOME),
            Map.entry(PROP_SO_KEEP_ALIVE, FWK_PROP_SO_KEEP_ALIVE),
            Map.entry(PROP_LEASE_SO_TIMEOUT, FWK_PROP_MONGO_LEASE_SO_TIMEOUT),
            Map.entry(PROP_UPDATE_LIMIT, FWK_PROP_UPDATE_LIMIT),
            Map.entry(PROP_MONGO_MAX_POOL_SIZE, FWK_PROP_MONGO_MAX_POOL_SIZE),
            Map.entry(PROP_MONGO_MIN_POOL_SIZE, FWK_PROP_MONGO_MIN_POOL_SIZE),
            Map.entry(PROP_MONGO_MAX_CONNECTING, FWK_PROP_MONGO_MAX_CONNECTING),
            Map.entry(PROP_MONGO_MAX_IDLE_TIME_MILLIS, FWK_PROP_MONGO_MAX_IDLE_TIME_MILLIS),
            Map.entry(PROP_MONGO_MAX_LIFE_TIME_MILLIS, FWK_PROP_MONGO_MAX_LIFE_TIME_MILLIS),
            Map.entry(PROP_MONGO_CONNECT_TIMEOUT_MILLIS, FWK_PROP_MONGO_CONNECT_TIMEOUT_MILLIS),
            Map.entry(PROP_MONGO_HEARTBEAT_FREQUENCY_MILLIS, FWK_PROP_MONGO_HEARTBEAT_FREQUENCY_MILLIS),
            Map.entry(PROP_MONGO_SERVER_SELECTION_TIMEOUT_MILLIS, FWK_PROP_MONGO_SERVER_SELECTION_TIMEOUT_MILLIS),
            Map.entry(PROP_MONGO_WAIT_QUEUE_TIMEOUT_MILLIS, FWK_PROP_MONGO_WAIT_QUEUE_TIMEOUT_MILLIS),
            Map.entry(PROP_MONGO_READ_TIMEOUT_MILLIS, FWK_PROP_MONGO_READ_TIMEOUT_MILLIS),
            Map.entry(PROP_MONGO_MIN_HEARTBEAT_FREQUENCY_MILLIS, FWK_PROP_MONGO_MIN_HEARTBEAT_FREQUENCY_MILLIS)
    );

    private DocumentNodeStoreServiceConfiguration() {
    }

    static Configuration create(ComponentContext context,
                                ConfigurationAdmin configurationAdmin,
                                Configuration preset,
                                Configuration configuration)
            throws IOException {
        return (Configuration) Proxy.newProxyInstance(
                DocumentNodeStoreServiceConfiguration.class.getClassLoader(),
                new Class[]{Configuration.class},
                new ConfigurationHandler(context, configurationAdmin, preset, configuration)
        );
    }

    private static String frameworkPropertyNameFor(String propertyName) {
        String fwkPropName = FWK_PROP_MAPPING.get(propertyName);
        if (fwkPropName == null) {
            fwkPropName = DEFAULT_FWK_PREFIX + propertyName;
        }
        return fwkPropName;
    }

    private static final class ConfigurationHandler implements InvocationHandler {

        private final ComponentContext context;

        /**
         * The preset configuration.
         */
        private final Configuration preset;

        /**
         * The configuration taking precedence over the preset.
         */
        private final Configuration configuration;

        private final Set<String> configurationKeys;

        ConfigurationHandler(ComponentContext context,
                             ConfigurationAdmin configurationAdmin,
                             Configuration preset,
                             Configuration configuration) throws IOException {
            this.context = requireNonNull(context);
            this.preset = requireNonNull(preset);
            this.configuration = requireNonNull(configuration);
            this.configurationKeys = getConfigurationKeys(requireNonNull(configurationAdmin));
        }

        private static Set<String> getConfigurationKeys(ConfigurationAdmin configurationAdmin)
                throws IOException {
            Set<String> keys = new HashSet<>();
            org.osgi.service.cm.Configuration c = configurationAdmin.getConfiguration(Configuration.PID);
            for (Object k : Collections.list(c.getProperties().keys())) {
                keys.add(k.toString());
            }
            return keys;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args)
                throws Throwable {
            String name = method.getName().replaceAll("_", ".");
            Configuration c;
            if (configurationKeys.contains(name)) {
                c = configuration;
            } else {
                c = preset;
            }
            Object value = method.invoke(c);
            // check if this is overridden by a framework property
            String frameworkProp = OsgiUtil.lookup(
                    context.getBundleContext(), frameworkPropertyNameFor(name));
            if (frameworkProp != null) {
                value = tryCoerce(frameworkProp, method.getReturnType(), value);
            }
            return value;
        }

        private Object tryCoerce(String value, Class<?> type, Object defaultValue) {
            Object obj;
            if (type == Boolean.class || type == boolean.class) {
                obj = Boolean.parseBoolean(value);
            } else if (type == Integer.class || type == int.class) {
                try {
                    obj = Integer.parseInt(value);
                } catch (NumberFormatException e) {
                    obj = defaultValue;
                }
            } else if (type == Long.class || type == long.class) {
                try {
                    obj = Long.parseLong(value);
                } catch (NumberFormatException e) {
                    obj = defaultValue;
                }
            } else if (type == String.class) {
                obj = value;
            } else if (type == String[].class) {
                obj = String.valueOf(value).split("::");
            } else {
                obj = defaultValue;
            }
            return obj;
        }
    }
}
