/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.commons.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.text.StringCharacterIterator;
import java.util.Arrays;
import java.util.Objects;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.iotdb.commons.file.SystemFileFactory;
import org.apache.iotdb.commons.utils.RetryUtils;
import org.apache.iotdb.commons.utils.WindowsOSUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileUtils.class);
    private static final int BUFFER_SIZE = 1024;
    private static final String RENAME_FILE_MESSAGE = "Renamed file {} to {} because it already exists in the target directory: {}";
    private static final String COPY_FILE_MESSAGE = "Copy file {} to {} because it already exists in the target directory: {}";
    private static final String ILLEGAL_PATH_MESSAGE = "The path cannot be '.', '..', './' or '.\\'. ";

    private FileUtils() {
    }

    public static boolean deleteFileIfExist(File file) {
        try {
            Files.deleteIfExists(file.toPath());
            return true;
        }
        catch (IOException e) {
            LOGGER.error(e.getMessage(), (Throwable)e);
            return false;
        }
    }

    public static void deleteFileOrDirectory(File file) {
        FileUtils.deleteFileOrDirectory(file, false);
    }

    public static void deleteFileOrDirectory(File file, boolean quietForNoSuchFile) {
        File[] files;
        if (file.isDirectory() && (files = file.listFiles()) != null) {
            for (File subfile : files) {
                FileUtils.deleteFileOrDirectory(subfile, quietForNoSuchFile);
            }
        }
        try {
            Files.delete(file.toPath());
        }
        catch (NoSuchFileException e) {
            if (!quietForNoSuchFile) {
                LOGGER.warn("{}: {}", new Object[]{e.getMessage(), Arrays.toString(file.list()), e});
            }
        }
        catch (DirectoryNotEmptyException e) {
            LOGGER.warn("{}: {}", new Object[]{e.getMessage(), Arrays.toString(file.list()), e});
        }
        catch (Exception e) {
            LOGGER.warn("{}: {}", new Object[]{e.getMessage(), file.getName(), e});
        }
    }

    public static void deleteFileOrDirectoryWithRetry(File file) {
        File[] files;
        if (file.isDirectory() && (files = file.listFiles()) != null) {
            for (File subfile : files) {
                FileUtils.deleteFileOrDirectoryWithRetry(subfile);
            }
        }
        try {
            RetryUtils.retryOnException(() -> {
                Files.delete(file.toPath());
                return null;
            });
        }
        catch (DirectoryNotEmptyException e) {
            LOGGER.warn("{}: {}", new Object[]{e.getMessage(), Arrays.toString(file.list()), e});
        }
        catch (Exception e) {
            LOGGER.warn("{}: {}", new Object[]{e.getMessage(), file.getName(), e});
        }
    }

    public static void deleteDirectoryAndEmptyParent(File folder) {
        FileUtils.deleteFileOrDirectory(folder);
        File parentFolder = folder.getParentFile();
        if (parentFolder.isDirectory() && Objects.requireNonNull(parentFolder.listFiles()).length == 0 && !parentFolder.delete()) {
            LOGGER.warn("Delete folder failed: {}", (Object)parentFolder.getAbsolutePath());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean copyDir(File sourceDir, File targetDir) throws IOException {
        if (!sourceDir.exists() || !sourceDir.isDirectory()) {
            LOGGER.error("Failed to copy folder, because source folder [{}] doesn't exist.", (Object)sourceDir.getAbsolutePath());
            return false;
        }
        if (!targetDir.exists() && !targetDir.mkdirs()) {
            Class<FileUtils> clazz = FileUtils.class;
            synchronized (FileUtils.class) {
                if (!targetDir.exists() && !targetDir.mkdirs()) {
                    LOGGER.error("Failed to copy folder, because failed to create target folder[{}].", (Object)targetDir.getAbsolutePath());
                    // ** MonitorExit[var2_2] (shouldn't be in output)
                    return false;
                }
                // ** MonitorExit[var2_2] (shouldn't be in output)
            }
        } else if (!targetDir.isDirectory()) {
            LOGGER.error("Failed to copy folder, because target folder [{}] already exist.", (Object)targetDir.getAbsolutePath());
            return false;
        }
        {
            File[] files;
            if ((files = sourceDir.listFiles()) == null || files.length == 0) {
                return true;
            }
            boolean result = true;
            for (File file : files) {
                if (!file.exists()) continue;
                File targetFile = new File(targetDir, file.getName());
                if (file.isDirectory()) {
                    result &= FileUtils.copyDir(file.getAbsoluteFile(), targetFile);
                    continue;
                }
                try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(file));
                     FileOutputStream fileOutputStream = new FileOutputStream(targetFile);
                     BufferedOutputStream out = new BufferedOutputStream(fileOutputStream);){
                    int size;
                    byte[] bytes = new byte[1024];
                    while ((size = in.read(bytes)) > 0) {
                        out.write(bytes, 0, size);
                    }
                    out.flush();
                    fileOutputStream.getFD().sync();
                }
                catch (IOException e) {
                    LOGGER.warn("get ioexception on file {}", (Object)file.getAbsolutePath(), (Object)e);
                    throw e;
                }
            }
            return result;
        }
    }

    public static long getDirSize(String path) {
        long sum = 0L;
        File file = SystemFileFactory.INSTANCE.getFile(path);
        if (file.isDirectory()) {
            String[] list;
            for (String item : list = file.list()) {
                String subPath = path + File.separator + item;
                sum += FileUtils.getDirSize(subPath);
            }
        } else {
            sum += file.length();
        }
        return sum;
    }

    public static void recursivelyDeleteFolder(String path) throws IOException {
        File file = new File(path);
        if (file.isDirectory()) {
            File[] files = file.listFiles();
            if (files == null || files.length == 0) {
                org.apache.commons.io.FileUtils.deleteDirectory((File)file);
            } else {
                for (File f : files) {
                    FileUtils.recursivelyDeleteFolder(f.getAbsolutePath());
                }
                org.apache.commons.io.FileUtils.deleteDirectory((File)file);
            }
        } else {
            org.apache.commons.io.FileUtils.delete((File)file);
        }
    }

    public static String addPrefix2FilePath(String prefix, String file) {
        if (!new File(file).isAbsolute() && prefix != null && prefix.length() > 0) {
            file = !prefix.endsWith(File.separator) ? prefix + File.separatorChar + file : prefix + file;
        }
        return file;
    }

    public static boolean moveFileSafe(File source, File target) {
        if (target.exists()) {
            LOGGER.info("won't move file again because target file already exists: {}", (Object)target.getAbsolutePath());
            LOGGER.info("you may manually delete source file if necessary: {}", (Object)source.getAbsolutePath());
            return true;
        }
        String fromTo = String.format("from %s to %s", source.getAbsolutePath(), target.getAbsolutePath());
        LOGGER.info("start to move file, {}", (Object)fromTo);
        File unfinishedTarget = new File(target.getAbsolutePath() + ".unfinished");
        try {
            if (unfinishedTarget.exists()) {
                if (unfinishedTarget.isFile()) {
                    org.apache.commons.io.FileUtils.delete((File)unfinishedTarget);
                } else {
                    FileUtils.recursivelyDeleteFolder(unfinishedTarget.getAbsolutePath());
                }
            }
        }
        catch (IOException e) {
            LOGGER.error("delete unfinished target file failed: {}", (Object)unfinishedTarget.getAbsolutePath(), (Object)e);
            return false;
        }
        LOGGER.info("unfinished target file which was created last time has been deleted: {}", (Object)unfinishedTarget.getAbsolutePath());
        try {
            if (source.isDirectory()) {
                if (!FileUtils.copyDir(source, unfinishedTarget)) {
                    LOGGER.error("file copy fail");
                    return false;
                }
            } else {
                org.apache.commons.io.FileUtils.copyFile((File)source, (File)unfinishedTarget);
            }
        }
        catch (IOException e) {
            LOGGER.error("file copy fail", (Throwable)e);
            return false;
        }
        if (!unfinishedTarget.renameTo(target)) {
            LOGGER.error("file rename fail");
            return false;
        }
        try {
            if (source.isDirectory()) {
                FileUtils.recursivelyDeleteFolder(source.getAbsolutePath());
            } else {
                org.apache.commons.io.FileUtils.delete((File)source);
            }
        }
        catch (IOException e) {
            LOGGER.error("delete source file fail: {}", (Object)source.getAbsolutePath(), (Object)e);
        }
        LOGGER.info("move file success, {}", (Object)fromTo);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static File createHardLink(File sourceFile, File hardlink) throws IOException {
        if (!hardlink.getParentFile().exists() && !hardlink.getParentFile().mkdirs()) {
            Class<FileUtils> clazz = FileUtils.class;
            // MONITORENTER : org.apache.iotdb.commons.utils.FileUtils.class
            if (!hardlink.getParentFile().exists() && !hardlink.getParentFile().mkdirs()) {
                throw new IOException(String.format("failed to create hardlink %s for file %s: failed to create parent dir %s", hardlink.getPath(), sourceFile.getPath(), hardlink.getParentFile().getPath()));
            }
            // MONITOREXIT : clazz
        }
        Path sourcePath = FileSystems.getDefault().getPath(sourceFile.getAbsolutePath(), new String[0]);
        Path linkPath = FileSystems.getDefault().getPath(hardlink.getAbsolutePath(), new String[0]);
        try {
            Files.createLink(linkPath, sourcePath);
            return hardlink;
        }
        catch (FileAlreadyExistsException fileAlreadyExistsException) {
            if (FileUtils.haveSameMD5(sourceFile, hardlink)) {
                LOGGER.warn("Hardlink {} already exists, will not create it again. Source file: {}", (Object)hardlink.getAbsolutePath(), (Object)sourceFile.getAbsolutePath());
                return hardlink;
            }
            LOGGER.warn("Hardlink {} already exists but does not match source file {}, will try create it again.", (Object)hardlink.getAbsolutePath(), (Object)sourceFile.getAbsolutePath());
            FileUtils.deleteFileIfExist(hardlink);
            try {
                Files.createLink(linkPath, sourcePath);
                return hardlink;
            }
            catch (Exception e) {
                FileUtils.deleteFileIfExist(linkPath.toFile());
                LOGGER.error("Failed to create hardlink {} for file {}: {}", new Object[]{hardlink.getAbsolutePath(), sourceFile.getAbsolutePath(), e.getMessage(), e});
                throw e;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     * Converted monitor instructions to comments
     * Lifted jumps to return sites
     */
    public static File copyFile(File sourceFile, File targetFile) throws IOException {
        if (!targetFile.getParentFile().exists() && !targetFile.getParentFile().mkdirs()) {
            Class<FileUtils> clazz = FileUtils.class;
            // MONITORENTER : org.apache.iotdb.commons.utils.FileUtils.class
            if (!targetFile.getParentFile().exists() && !targetFile.getParentFile().mkdirs()) {
                throw new IOException(String.format("failed to copy file %s to %s: failed to create parent dir %s", sourceFile.getPath(), targetFile.getPath(), targetFile.getParentFile().getPath()));
            }
            // MONITOREXIT : clazz
        }
        Files.copy(sourceFile.toPath(), targetFile.toPath(), new CopyOption[0]);
        return targetFile;
    }

    public static String humanReadableByteCountSI(long bytes) {
        if (-1000L < bytes && bytes < 1000L) {
            return bytes + " B";
        }
        StringCharacterIterator ci = new StringCharacterIterator("KMGTPE");
        while (bytes <= -999950L || bytes >= 999950L) {
            bytes /= 1000L;
            ci.next();
        }
        return String.format("%.2f %cB", (double)bytes / 1000.0, Character.valueOf(ci.current()));
    }

    public static void moveFileWithMD5Check(File sourceFile, File targetDir) throws IOException {
        String sourceFileName = sourceFile.getName();
        File targetFile = new File(targetDir, sourceFileName);
        if (targetFile.exists()) {
            FileUtils.moveFile(sourceFile, targetDir);
        } else {
            org.apache.commons.io.FileUtils.moveFileToDirectory((File)sourceFile, (File)targetDir, (boolean)true);
        }
    }

    private static void moveFile(File sourceFile, File targetDir) throws IOException {
        String existsFileMD5;
        String sourceFileMD5;
        long existsFileSize;
        String sourceFileName = sourceFile.getName();
        File exitsFile = new File(targetDir, sourceFileName);
        long sourceFileSize = sourceFile.length();
        if (sourceFileSize != (existsFileSize = exitsFile.length())) {
            File file = FileUtils.renameWithSize(sourceFile, sourceFileSize, targetDir);
            if (!file.exists()) {
                FileUtils.moveFileRename(sourceFile, file);
            }
            return;
        }
        try (FileInputStream is1 = new FileInputStream(sourceFile);
             FileInputStream is2 = new FileInputStream(exitsFile);){
            sourceFileMD5 = DigestUtils.md5Hex((InputStream)is1);
            existsFileMD5 = DigestUtils.md5Hex((InputStream)is2);
        }
        if (sourceFileMD5.equals(existsFileMD5)) {
            org.apache.commons.io.FileUtils.forceDelete((File)sourceFile);
            LOGGER.info("Deleted the file {} because it already exists in the target directory: {}", (Object)sourceFile.getName(), (Object)targetDir.getAbsolutePath());
        } else {
            File file = FileUtils.renameWithMD5(sourceFile, sourceFileMD5, targetDir);
            FileUtils.moveFileRename(sourceFile, file);
        }
    }

    public static void copyFileWithMD5Check(File sourceFile, File targetDir) throws IOException {
        String sourceFileName = sourceFile.getName();
        File targetFile = new File(targetDir, sourceFileName);
        if (targetFile.exists()) {
            FileUtils.copyFileWithMD5(sourceFile, targetDir);
        } else {
            try {
                Files.createDirectories(targetDir.toPath(), new FileAttribute[0]);
            }
            catch (IOException e) {
                LOGGER.warn("failed to create target directory: {}", (Object)targetDir.getAbsolutePath());
                throw e;
            }
            Files.copy(sourceFile.toPath(), targetFile.toPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
        }
    }

    private static File renameWithMD5(File sourceFile, String sourceFileMD5, File targetDir) throws IOException {
        String sourceFileBaseName = FilenameUtils.getBaseName((String)sourceFile.getName());
        String sourceFileExtension = FilenameUtils.getExtension((String)sourceFile.getName());
        String targetFileName = sourceFileBaseName + "-" + sourceFileMD5.substring(0, 16) + "." + sourceFileExtension;
        return new File(targetDir, targetFileName);
    }

    private static void copyFileWithMD5(File sourceFile, File targetDir) throws IOException {
        String exitsFileMD5;
        String sourceFileMD5;
        long exitsFileSize;
        String sourceFileName = sourceFile.getName();
        File exitsFile = new File(targetDir, sourceFileName);
        long sourceFileSize = sourceFile.length();
        if (sourceFileSize != (exitsFileSize = exitsFile.length())) {
            File file = FileUtils.renameWithSize(sourceFile, sourceFileSize, targetDir);
            if (!file.exists()) {
                FileUtils.copyFileRename(sourceFile, file);
            }
            return;
        }
        try (FileInputStream is1 = new FileInputStream(sourceFile);
             FileInputStream is2 = new FileInputStream(exitsFile);){
            sourceFileMD5 = DigestUtils.md5Hex((InputStream)is1);
            exitsFileMD5 = DigestUtils.md5Hex((InputStream)is2);
        }
        if (sourceFileMD5.equals(exitsFileMD5)) {
            return;
        }
        File file = FileUtils.renameWithMD5(sourceFile, sourceFileMD5, targetDir);
        if (!file.exists()) {
            FileUtils.copyFileRename(sourceFile, file);
        }
    }

    private static File renameWithSize(File sourceFile, long sourceFileSize, File targetDir) {
        String sourceFileBaseName = FilenameUtils.getBaseName((String)sourceFile.getName());
        String sourceFileExtension = FilenameUtils.getExtension((String)sourceFile.getName());
        String newFileName = String.format("%s_%s_%s.%s", sourceFileBaseName, sourceFileSize, System.currentTimeMillis(), sourceFileExtension);
        return new File(targetDir, newFileName);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static boolean haveSameMD5(File file1, File file2) {
        try (InputStream is1 = Files.newInputStream(file1.toPath(), new OpenOption[0]);){
            boolean bl;
            block14: {
                InputStream is2 = Files.newInputStream(file2.toPath(), new OpenOption[0]);
                try {
                    bl = DigestUtils.md5Hex((InputStream)is1).equals(DigestUtils.md5Hex((InputStream)is2));
                    if (is2 == null) break block14;
                }
                catch (Throwable throwable) {
                    if (is2 != null) {
                        try {
                            is2.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                is2.close();
            }
            return bl;
        }
        catch (Exception e) {
            return false;
        }
    }

    private static void moveFileRename(File sourceFile, File targetFile) throws IOException {
        org.apache.commons.io.FileUtils.moveFile((File)sourceFile, (File)targetFile, (CopyOption[])new CopyOption[]{StandardCopyOption.REPLACE_EXISTING});
        LOGGER.info(RENAME_FILE_MESSAGE, new Object[]{sourceFile.getName(), targetFile.getName(), targetFile.getParentFile().getAbsolutePath()});
    }

    private static void copyFileRename(File sourceFile, File targetFile) throws IOException {
        Files.copy(sourceFile.toPath(), targetFile.toPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
        LOGGER.info(COPY_FILE_MESSAGE, new Object[]{sourceFile.getName(), targetFile, targetFile.getParentFile().getAbsolutePath()});
    }

    public static String getIllegalError4Directory(String path) {
        if (path.equals(".") || path.equals("..") || path.contains("/") || path.contains("\\")) {
            return ILLEGAL_PATH_MESSAGE;
        }
        if (!WindowsOSUtils.isLegalPathSegment4Windows(path)) {
            return WindowsOSUtils.OS_SEGMENT_ERROR;
        }
        return null;
    }
}

